<?php

namespace App\Http\Controllers\API\Auth\Customer;

use App\Domain\User\Actions\CreateOrUpdateFcmToken;
use App\Domain\User\Models\FcmToken;
use App\Http\Controllers\API\ApiController;
use App\Http\Resources\API\User\UserResource;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Laravel\Sanctum\Sanctum;

class EditUserController extends ApiController
{
    public function __construct()
    {
        $this->middleware('auth:sanctum');
    }

    public function __invoke(Request $request)
    {
        $user = auth('sanctum')->user();
        $validated = $this->validate(request(), [
            'name' => 'sometimes|nullable|string|max:100',
            'first_name' => 'sometimes|nullable|string|max:100',
            'last_name' => 'sometimes|nullable|string|max:100',
            'city_id' => 'sometimes|nullable|integer|exists:cities,id',
            'company_id' => 'sometimes|nullable|integer|exists:companies,id',
            'area_id' => 'sometimes|nullable|integer|exists:areas,id,deleted_at,NULL',
            'identification' => "sometimes|nullable|digits:10",
        ]);

        $inputs = collect($validated)->filter()->all();

        $user->fill($inputs);
        $user->save();

        if (request('business_card')) {
            $user->clearMediaCollection('business_card');
            $user->addMedia(request('business_card'))->toMediaCollection('business_card');
        }
        if (request('license_image')) {
            $user->clearMediaCollection('license_image');
            $user->addMedia(request('license_image'))->toMediaCollection('license_image');
        }
        if (request('identification_image')) {
            $user->clearMediaCollection('identification_image');
            $user->addMedia(request('identification_image'))->toMediaCollection('identification_image');
        }
        if (request('avatar')) {
            $user->clearMediaCollection('avatar');
            $user->addMedia(request('avatar'))->toMediaCollection('avatar');
        }

        $response = [
            'user' => new UserResource($user),
            'token' => $request->bearerToken(),
        ];

        return $this->sendResponse(
            $response,
            __('User retrieved Successfully')
        );
    }


    /**
     * @return \Illuminate\Http\JsonResponse
     */
    public function getProfile()
    {
        $user = request()->user();
        $token = request()->bearerToken();

        $response = [
            'user' => new UserResource($user),
            'token' => $token,
        ];
        return $this->sendResponse(
            $response,
            __('User retrieved Successfully')
        );
    }

    public function updateToken()
    {
        $this->validate(request(), [
            'fcm_type' => 'required|string|in:android,ios',
            'fcm_token' => 'required|string',
        ]);
        $user = auth('sanctum')->user();
        if (request('fcm_type') && request('fcm_token')) {
            (new CreateOrUpdateFcmToken())($user, request('fcm_token'), request('fcm_type'));
        }
        return $this->sendSuccess(
            __('Token Updated')
        );
    }

    public function updateLocation()
    {

        $this->validate(request(), [
            'lat' => 'required|string',
            'lng' => 'required|string',
            'address' => 'required|string',
        ]);
        $user = auth('sanctum')->user();

        $user->update([
            'lat' => request('lat'),
            'lng' => request('lng'),
            'address' => request('address'),
            'last_updated_location' => Carbon::now(),
        ]);
        return $this->sendSuccess(
            __('Location Updated')
        );
    }

    public function deleteAccount()
    {
        $validated = $this->validate(request(), [
            'fcm_type' => 'sometimes|in:web,ios,android',
        ]);
        $user = auth('sanctum')->user();
        if (!$user) {
            return $this->sendError(
                __('Invalid Token')
            );
        }
        FcmToken::reset(auth('sanctum')->user(), 'ios');
        FcmToken::reset(auth('sanctum')->user(), 'android');
        $model = Sanctum::$personalAccessTokenModel;

        if ($accessToken = $model::findToken(request()->bearerToken())) {
            $accessToken->delete();
        }

        $user->update([
            'phone' => rand(1, 999999) . hash('sha256', $user->phone),
            'old_phone' => $user->phone,
        ]);
        $user->delete();
        return $this->sendSuccess(
            __('User Deleted Successfully')
        );
    }
}
