<?php

namespace App\Http\Controllers\API\Auth;

use App\Domain\Core\Models\Administration\User;
use App\Domain\Core\Models\Otp;
use App\Domain\User\Actions\ArmadarSms;
use App\Domain\User\Actions\CreateOrUpdateFcmToken;
use App\Domain\User\Models\FcmToken;
use App\Http\Controllers\API\ApiController;
use App\Http\Resources\API\User\UserResource;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Laravel\Sanctum\Sanctum;

class LoginOrRegisterController extends ApiController
{
    public function __invoke(Request $request)
    {
        $validated = $this->AuthValidation($request);

        $otpCode = Arr::pull($validated, 'otp_code');
        $user = User::wherePhone($request->phone)->first();
        // Check user roles and user-type
        $errorResponse = $this->checkUserRolesAndType($user, $request->header('user-type'));
        if ($errorResponse) {
            return $errorResponse;
        }
        $checkCode = (new ArmadarSms())->CheckCode($otpCode, $request->phone);
        if (!$checkCode) {
            return $this->sendError(__('Code Is Incorrect'));
        }
        if ($user) {
            // now you can log in
            $token = $user->createToken('User Login')->plainTextToken;
            $response = [
                'user' => new UserResource($user),
                'token' => $token,
            ];
        } else {
            // register the new user
            DB::beginTransaction();
            try {
                $license_image = Arr::pull($validated, 'license_image');
                $identification_image = Arr::pull($validated, 'identification_image');
                $user = User::create($validated);
                $user->assignRole($request->header('user-type') ?? 'user');
                if (request('fcm_type') && request('fcm_token')) {
                    (new CreateOrUpdateFcmToken())($user, request('fcm_token'), request('fcm_type'));
                }
                DB::commit();
            } catch (\Exception $e) {
                DB::rollback();
                return $this->sendError(__('something happened wrong , please contact support'));
            }
            $token = $user->createToken('User Register')->plainTextToken;
            $response = [
                'user' => new UserResource($user),
                'token' => $token,
            ];
        }

        if (request('fcm_type') && request('fcm_token')) {
            (new CreateOrUpdateFcmToken())($user, request('fcm_token'), request('fcm_type'));
        }
        // add media for user or driver
        $this->addMediaForUser($user);
        // delete old otp
        Otp::where('phone', $request->phone)->delete();
        return $this->sendResponse(
            $response,
            __('User retrieved Successfully')
        );
    }

    public function logOut(Request $request)
    {
        $validated = $this->validate($request, [
            'fcm_type' => 'sometimes|in:web,ios,android',
        ]);
        $user = auth('sanctum')->user();
        if (!$user) {
            return $this->sendError(
                __('Invalid Token')
            );
        }
        FcmToken::reset(auth('sanctum')->user(), $validated['fcm_type'] ?? 'ios');
        $model = Sanctum::$personalAccessTokenModel;

        if ($accessToken = $model::findToken($request->bearerToken())) {
            $accessToken->delete();
        }
        return $this->sendSuccess(
            __('User Logged Out Successfully')
        );

    }

    public function sendOtp()
    {
        $user = User::wherePhone(request('phone'))->first();
        // Check user roles and user-type
        $errorResponse = $this->checkUserRolesAndType($user, request()->header('user-type'));
        if ($errorResponse) {
            return $errorResponse;
        }
        $sendOtp = (new ArmadarSms())->send(request('phone'));
        if ($sendOtp['status']) {
            return $this->sendResponse([
                'registered' => (boolean)$user
            ], __('SMS Has Been Sent To Your Phone'));
            return $this->sendSuccess(__('SMS Has Been Sent To Your Phone'));
        }
        return $this->sendError($sendOtp['error']);
    }

    public function CheckCode(Request $request)
    {
        $this->validate($request, [
            'phone' => 'required|string',
            'code' => 'required|string',
        ]);

        $checkCode = (new ArmadarSms())->CheckCode($request->code, $request->phone);

        if (!$checkCode) {
            return $this->sendError(__('Code Is Incorrect'));
        }

        return $this->sendSuccess(
            __('Code Is Correct')
        );
    }

    /**
     * @param Request $request
     * @return array
     * @throws \Illuminate\Validation\ValidationException
     */
    protected function AuthValidation(Request $request): array
    {
        if ($request->header('user-type') == 'user') {
            $validated = $this->validate($request, [
                'phone' => 'required|string|max:15',
                'first_name' => 'required_if:registered,false|string|max:100',
                'last_name' => 'required_if:registered,false|string|max:100',
                'city_id' => 'required_if:registered,false|integer|exists:cities,id,deleted_at,NULL',
                'area_id' => 'required_if:registered,false|integer|exists:areas,id,deleted_at,NULL',
                'is_broker' => 'sometimes|nullable',
                'otp_code' => 'required|string',
                'company_id' => 'sometimes|nullable|integer|exists:companies,id,deleted_at,NULL',
            ]);
        } else {
            $validated = $this->validate($request, [
                'phone' => 'required|string|max:15',
                'first_name' => 'required_if:registered,false|string|max:100',
                'last_name' => 'required_if:registered,false|string|max:100',
                'city_id' => 'required_if:registered,false|integer|exists:cities,id,deleted_at,NULL',
                'area_id' => 'required_if:registered,false|integer|exists:areas,id,deleted_at,NULL',
                'identification' => "required_if:registered,false|digits:10",
                'license_image' => 'required_if:registered,false|image|max:10000',
                'identification_image' => 'required_if:registered,false|image|max:10000',
                'otp_code' => 'required|string',
            ]);
        }
        return $validated;
    }

    /**
     * @param $user
     * @return void
     */
    protected function addMediaForUser($user): void
    {
        if (request('business_card')) {
            $user->clearMediaCollection('business_card');
            $user->addMedia(request('business_card'))->toMediaCollection('business_card');
        }
        if (request('license_image')) {
            $user->clearMediaCollection('license_image');
            $user->addMedia(request('license_image'))->toMediaCollection('license_image');
        }
        if (request('identification_image')) {
            $user->clearMediaCollection('identification_image');
            $user->addMedia(request('identification_image'))->toMediaCollection('identification_image');
        }
    }

    private function checkUserRolesAndType($user, $userType)
    {
        if ($user?->hasRole('user') && $userType == 'driver') {
            return $this->sendError(__("Sorry you can't login in this app, You Are Customer, please use other application"));
        }
        if ($user?->hasRole('driver') && $userType == 'user') {
            return $this->sendError(__("Sorry you can't login in this application, You Are Driver, please use other application"));
        }
        return null; // No error response
    }
}
