<?php

namespace App\Http\Controllers\API\Auth;

use App\Domain\Core\Models\Administration\User;
use App\Domain\User\Models\FcmToken;
use App\Http\Controllers\API\ApiController;
use App\Http\Resources\API\Core\CompanyResource;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Password;
use Laravel\Sanctum\Sanctum;


class LoginController extends ApiController
{
    public function logOut(Request $request)
    {
        $validated = $this->validate($request, [
            'fcm_type' => 'sometimes|in:web,ios,android',
        ]);
        $user = auth('sanctum')->user();
        if (!$user) {
            return $this->sendError(
                __('Invalid Token')
            );
        }

        FcmToken::reset(auth('sanctum')->user(), $validated['fcm_type'] ?? 'ios');
        $model = Sanctum::$personalAccessTokenModel;

        if ($accessToken = $model::findToken($request->bearerToken())) {
            $accessToken->delete();
        }
        return $this->sendSuccess(
            __('User Logged Out Successfully')
        );

    }


    public function login(Request $request)
    {
        $request->validate([
            'phone_or_email' => ['required', 'string'],
            'password' => ['required', 'string'],
        ]);

        $credentials = $request->only('phone_or_email', 'password');

        if (Auth::attempt(['email' => $credentials['phone_or_email'], 'password' => $credentials['password']]) ||
            Auth::attempt(['phone' => $credentials['phone_or_email'], 'password' => $credentials['password']])
        ) {
            $user = Auth::user();
            $token = $user->createToken('User Login')->plainTextToken;

            $response = [
                'user' => new CompanyResource($user),
                'token' => $token,
                'redirect_url' => route('dashboard.home', ['token' => $token])
            ];
            return $this->sendResponse(
                $response,
                __('User retrieved Successfully')
            );
        }
        return $this->sendError(__('Invalid Credentials'));
    }

    public function register(Request $request)
    {
        $data = $request->validate([
            'name' => ['required', 'string'],
            'email' => ['required', 'string', 'email', 'unique:users'],
            'phone' => ['required', 'string', 'unique:users'],
            'password' => ['required', 'string', 'min:8'],
        ]);

        $user = User::create($data);
        $user->assignRole('company');
        $token = $user->createToken('User Registration')->plainTextToken;

        $response = [
            'user' => new CompanyResource($user),
            'token' => $token,
            'redirect_url' => route('dashboard.home', ['token' => $token])
        ];
        return $this->sendResponse(
            $response,
            __('User retrieved Successfully')
        );
    }

    public function sendResetLink(Request $request)
    {
        $request->validate([
            'email' => ['required', 'string'],
        ]);

        $user = User::where('email', $request->email)
            ->first();

        if (!$user) {
            return $this->sendError(__('User not found'));
        }

        $response = Password::sendResetLink(
            $request->only('email')
        );

        return $response == Password::RESET_LINK_SENT
            ? $this->sendSuccess(__('Reset link sent to your email'))
            : $this->sendError(__('Unable to send reset link'));
    }
}
