<?php

namespace App\Domain\Blog\Models;

use App\Domain\Core\Models\Administration\User;
use App\Support\Concerns\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Spatie\Image\Manipulations;
use Spatie\ImageOptimizer\OptimizerChainFactory;
use Spatie\ImageOptimizer\Optimizers\Cwebp;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\MediaLibrary\MediaCollections\Models\Media;
use Spatie\Translatable\HasTranslations;

class Post extends Model implements HasMedia
{
    use HasFactory, InteractsWithMedia;
    //use HasTranslations;

    protected $guarded = [];

    protected $casts = [
        'published_at' => 'datetime',
    ];

    protected $translatable = [
        'title',
        'slug',
        'content',
        'seo_title',
        'seo_description',
        'seo_keywords',
        'canonical_url',
        'og_title',
        'og_description',
        'twitter_title',
        'twitter_description', ];

    public function category()
    {
        return $this->belongsTo(PostCategory::class);
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function registerMediaConversions(Media $media = null): void
    {
        // Create the optimizer chain
        $optimizerChain = OptimizerChainFactory::create();

        // Add Cwebp optimizer to the chain with parameters
        $optimizerChain->addOptimizer(new Cwebp(['-q' => 50]));

        // Watermarked conversion with WebP format
        $this->addMediaConversion('watermarked')
            ->format('webp')
            ->watermarkPosition(Manipulations::POSITION_TOP_RIGHT)
            ->watermark(public_path('frontend/dist/images/logo.png'))
            ->watermarkWidth(150)    // Set the width of the watermark to 150 pixels
            ->watermarkHeight(150)   // Set the height of the watermark to 150 pixels
            ->watermarkOpacity(70)
            ->optimize($optimizerChain); // Use the OptimizerChain for optimization

        // Resized conversion with WebP format
        $this->addMediaConversion('resized')
            ->format('webp') // Set the format to WebP
            ->fit(Manipulations::FIT_CROP, 537, 408) // Resize to 537x408 using crop
            ->watermark(public_path('frontend/dist/images/logo.png')) // Optional watermark
            ->watermarkPosition(Manipulations::POSITION_TOP_RIGHT)
            ->watermarkWidth(150)
            ->watermarkHeight(150)
            ->watermarkOpacity(70)
            ->nonQueued(); // Ensures the conversion happens immediately
    }



    public function getOgImageAttribute()
    {
        return ! empty($this->getFirstMediaUrl('og_image', 'watermarked')) ? $this->getFirstMediaUrl('og_image', 'watermarked') :
            null;
    }

    public function getMainImageAttribute()
    {
        return ! empty($this->getFirstMediaUrl('main_image', 'watermarked')) ? $this->getFirstMediaUrl('main_image', 'watermarked') :
            'https://ui-avatars.com/api/?background=0D8ABC&color=fff&name='.$this->title;
    }

    public function resizedMainImage()
    {
        return ! empty($this->getFirstMediaUrl('main_image', 'resized')) ? $this->getFirstMediaUrl('main_image', 'resized') :
            'https://ui-avatars.com/api/?background=0D8ABC&color=fff&name='.$this->title;
    }

    public function getTwitterImageAttribute()
    {
        return ! empty($this->getFirstMediaUrl('twitter_image', 'watermarked')) ? $this->getFirstMediaUrl('twitter_image', 'watermarked') :
            null;
    }
}
